#include <casa/aips.h>
#include <casa/iostream.h>
#include <casa/iomanip.h>

#include <measures/Measures.h>
#include <measures/Measures/MeasFrame.h>
#include <measures/Measures/MCEpoch.h>
#include <measures/Measures/MCDirection.h>
#include <measures/Measures/MCPosition.h>
#include <measures/Measures/MCFrequency.h>
#include <measures/Measures/MCRadialVelocity.h>
#include <measures/Measures/MCDoppler.h>


#include <casa/Exceptions/Error.h>

using namespace casa;

int main() {
  try {
    // Observed peak topocentric frequencies of the maser at 3 given epochs
    Vector<Double> freqs(3);
    freqs(0) = 4.76503417969;
    freqs(1) = 4.76469824219;
    freqs(2) = 4.76493066406;
    // epochs of observations (mjd)
    Vector<Double> epochs(3);
    epochs(0) = 54468.493923611109;
    epochs(1) = 54542.278935185182;
    epochs(2) = 54614.149143518516;

    // rest frequency in Hz (needed for velocity-based conversion)
    const Double restfreq = 4.765562e9;
    
    // buffer for LSRK frequencies and velocity
    Vector<Double> outfreqs(3);
    Vector<Double> outvels(3);
    Vector<Double> outfreqs_viavel(3);
    
    
    // The frequency axis increment in Hz, given here just to show the error in terms of the 
    // spectral channel width. The main reason for this is to give a right feeling how big the 
    // error is. All conversions are done for single frequencies. 
    Double deltaf = 976.5625;

    // Hobart using VLBI position
    MPosition obs(MVPosition(-3950236.7341,
                             2522347.553,
                             -4311562.5434),MPosition::ITRF);

    // Monr2 maser at 4.7 GHz checked against literature
    MDirection coord(Quantity(1.6048229750035694,"rad"),
                     Quantity(-0.11139370025381365,"rad"),
                     MDirection::J2000);
    
    for (uInt i=0; i < epochs.nelements(); ++i) {
      //UTC epoch varies
      MEpoch epo0(Quantity(epochs(i),"d"));
      MeasFrame frame(obs,coord,epo0);
      MFrequency mf0(Quantity(freqs(i), "GHz"), MFrequency::TOPO);
      
      // convert TOPO frequency to LSRK 
      MFrequency mfout0 = MFrequency::Convert(mf0, 
                                              MFrequency::Ref(MFrequency::LSRK,
                                                              frame))();
      outfreqs[i] = mfout0.get(Unit("Hz")).getValue();
      
      // convert TOPO frequency to TOPO velocity
      MRadialVelocity mv0 = MRadialVelocity::fromDoppler(mf0.toDoppler(restfreq),MRadialVelocity::TOPO);
      
      // convert TOPO velocity to LSRK velocity
      MRadialVelocity mvout0 = MRadialVelocity::Convert(mv0, 
                                              MRadialVelocity::Ref(MRadialVelocity::LSRK, frame))();
      outvels[i] = mvout0.get("km/s").getValue();
    
      // convert LSRK velocity to LSRK frequency
      mfout0 = MFrequency::fromDoppler(mvout0.toDoppler(), restfreq, MFrequency::LSRK);
      outfreqs_viavel[i] = mfout0.get("Hz").getValue();
        
    }
    cout << "In freqs:  " << setprecision(10) << freqs << endl;
    cout << "Out freqs: " << setprecision(10) << outfreqs << endl;
    cout << "... via velocity: " << setprecision(10) << outfreqs_viavel << endl;
    cout << "Out vel: " << setprecision(10) << outvels << endl;
    
    
    // This should be < 1.0
    cout << "Difference 2-0" << endl; 
    cout << "LSR via freq (chan): " << (outfreqs[2] - outfreqs[0])/deltaf << endl;
    cout << "LSR via vel (chan): " << (outfreqs_viavel[2] - outfreqs_viavel[0])/deltaf << endl;
    cout << "LSR velocities (km/s): " << (outvels[2] - outvels[0]) << endl;  
    cout << "TOPO (in):  " << (freqs[2] - freqs[0])*1e9/deltaf << endl;
    cout << "Difference 1-0" << endl;
    cout << "LSR via freq (chan): " << (outfreqs[1] - outfreqs[0])/deltaf << endl;
    cout << "LSR via vel (chan): " << (outfreqs_viavel[1] - outfreqs_viavel[0])/deltaf << endl;
    cout << "LSR velocities (km/s): " << (outvels[1] - outvels[0]) << endl;  
    cout << "TOPO (in):  " << (freqs[1] - freqs[0])*1e9/deltaf << endl;
  } catch (AipsError &e) {
    cout << e.what() << endl;
  }
  return 0;
}
