#include <casa/aips.h>
#include <casa/iostream.h>
#include <casa/iomanip.h>

#include <measures/Measures.h>
#include <measures/Measures/MeasFrame.h>
#include <measures/Measures/MCEpoch.h>
#include <measures/Measures/MCDirection.h>
#include <measures/Measures/MCPosition.h>
#include <measures/Measures/MCFrequency.h>

#include <casa/Exceptions/Error.h>

using namespace casa;

int main() {
  try {
    // observation frequencies at the peak of the maser for three epochs
    Vector<Double> freqs(3);
    freqs(0) = 4.76503417969;
    freqs(1) = 4.76469824219;
    freqs(2) = 4.76493066406;
    // the corresponding epochs
    Vector<Double> epochs(3);
    epochs(0) = 54468.493923611109;
    epochs(1) = 54542.278935185182;
    epochs(2) = 54614.149143518516;

    Vector<Double> outfreqs(3);

    // Hobart using VLBI position
    MPosition obs(MVPosition(-3950236.7341,
                             2522347.553,
                             -4311562.5434),MPosition::ITRF);

    // Monr2 maser at 4.7 GHz checked against literature
    MDirection coord(Quantity(1.6048229750035694,"rad"),
                     Quantity(-0.11139370025381365,"rad"),
                     MDirection::J2000);
    for (uInt i=0; i < freqs.nelements(); ++i) {
      //UTC epoch varies
      MEpoch epo0(Quantity(epochs(i),"d"));
      MeasFrame frame(obs,coord,epo0);
      MFrequency mf0(Quantity(freqs(i), "GHz"), MFrequency::TOPO);
      MFrequency mfout0 = MFrequency::Convert(mf0, 
                                              MFrequency::Ref(MFrequency::LSRK,
                                                              frame))();
      outfreqs[i] = mfout0.get(Unit("Hz")).getValue();
    }
    cout << setprecision(10) << outfreqs << endl;
    // The frequency axis increment 
    Double deltaf = 976.5625;
    // This should be < 1.0 
    cout << (outfreqs[2] - outfreqs[0])/deltaf << endl;
  } catch (AipsError &e) {
    cout << e.what() << endl;
  }
  return 0;
}
